<?php

namespace App\Http\Controllers;

use App\Models\BranchModel;
use App\Models\ExpenseCategory;
use App\Models\ExpensesModel;
use App\Models\InvoiceEmiDtlsModel;
use App\Models\InvoiceEmiModel;
use App\Models\InvoiceModel;
use App\Models\LoyaltyTransactionModel;
use App\Models\PaymentStatusModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductModel;
use App\Models\PurchaseModel;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class ReportController extends Controller
{

    public function productFilter(Request $request)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $status = $request->input('status');

        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        if ($status === 'current') {
            $query = $isAdmin
                ? ProductModel::with('branch')
                : ProductModel::with('branch')->where('branch_id', session('branch_id'));

            $results = $query->get();
        } else {
            $query = $isAdmin
                ? ProductHistoryModel::with('branch')
                : ProductHistoryModel::with('branch')->where('branch_id', session('branch_id'));

            if ($start_date) {
                $query->whereDate('created_at', '>=', $start_date);
            }

            if ($end_date) {
                $query->whereDate('created_at', '<=', $end_date);
            }

            if ($status) {
                $query->where('action', $status);
            }

            $results = $query->with('user')->get();
        }

        return view('reports.productReport', compact('results', 'status'));
    }



    public function invoiceFilter(Request $request)
    {
        // Validate the incoming request for start and end dates
        $results = InvoiceModel::with('branch')->where('oprntl_flag', 'A')->where('active_flag', 'A')->get();
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'nullable|in:Y,N',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $status = $request->input('status');

        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $query = InvoiceModel::query();

        if ($start_date) {
            $query->where('created_at', '>=', $start_date);
        }

        if ($end_date) {
            $query->where('created_at', '<=', $end_date);
        }

        if ($status) {
            $query->where('gst', '=', $status);
        }

        $results = $isAdmin ? $query->with('branch')->where('oprntl_flag', 'A')->where('active_flag', 'A')->get() : $query->with('branch')->where('oprntl_flag', 'A')->where('branch_id', session('branch_id'))->where('active_flag', 'A')->get();
        $statusList = DB::table('payment_status')->pluck('name', 'id');

        return view('reports.invoiceReport', compact('results', 'statusList'));
    }

    public function profitFilter(Request $request)
    {
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);
        $results = $isAdmin ? InvoiceModel::with(['invoiceItems.products', 'invoiceItems.invoiceItemTaxes', 'branch'])
            ->where('oprntl_flag', 'A')->where('active_flag', 'A')->get() : InvoiceModel::with(['invoiceItems.products', 'invoiceItems.invoiceItemTaxes', 'branch'])->where('branch_id', session('branch_id'))
                ->where('oprntl_flag', 'A')->where('active_flag', 'A')->get();
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $branch = BranchModel::where('id', session('branch_id'))->first();

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if ($start_date && $end_date) {
            $results = $results->filter(function ($invoice) use ($start_date, $end_date) {
                $invoiceDate = Carbon::parse($invoice->invoice_date);
                return $invoiceDate->between($start_date, $end_date);
            });
        }

        $profitDetails = [];
        $discount = 0;

        foreach ($results as $invoice) {
            $totalSellingPrice = 0;
            $totalMarginPrice = 0;
            $totalProfit = 0;
            $totalTaxAmount = 0;
            $redeem = 0;
            $productProfitDetails = [];
            if (isset($invoice->discount)) {
                $discount = $invoice->discount;
            }

            $points = LoyaltyTransactionModel::where('reference_invoice', $invoice->invoice_id)->where('type', 'redeem')->first();


            foreach ($invoice->invoiceItems as $invoiceItem) {
                $product = $invoiceItem->products;

                $sellingPrice = $invoiceItem->price;
                $marginPrice = $invoiceItem->margin_price;
                $quantity = $invoiceItem->quantity;

                $profit = ($sellingPrice - $marginPrice) * $quantity;

                $taxAmount = 0;
                foreach ($invoiceItem->invoiceItemTaxes as $tax) {
                    $taxAmount += ($tax->tax / 100) * $sellingPrice * $quantity;
                }
                $totalSellingPrice += ($sellingPrice * $quantity) + $taxAmount;

                $totalMarginPrice += $marginPrice * $quantity;

                $totalTaxAmount += $taxAmount;

                if (!empty($points)) {
                    $redeem = max(floor($points->points * $branch->point_redemption), 2);
                }

                if ($invoice->discount_type == 1) {
                    $discount = $invoice->discount;
                } else {
                    $discount = $totalSellingPrice * ($invoice->discount / 100);
                }
                $productProfitDetails[] = [
                    'product_name' => $invoice->branch->native_name_flag == 'A' ? ($product->native_name ?? $product->name) : $product->name,
                    'selling_price' => $sellingPrice,
                    'margin_price' => $marginPrice,
                    'quantity' => $quantity,
                    'profit' => $profit,
                    'tax_amount' => $taxAmount,
                ];
            }
            // dump($totalTaxAmount);
            if (isset($discount)) {
                $totalSellingPrice = $totalSellingPrice - $discount;
            }
            if ($branch->loyalty_config == 'A' & !empty($redeem)) {
                $totalSellingPrice = $totalSellingPrice - $redeem;
            }

            $totalProfit = max(floor($totalSellingPrice - $totalMarginPrice), 2);
            $profitDetails[] = [
                'branch_name' => $invoice->branch->name ?? '',
                'invoice_id' => $invoice->invoice_id,
                'invoice_date' => $invoice->invoice_date,
                'total_selling_price' => floor($totalSellingPrice),
                'total_margin_price' => $totalMarginPrice,
                'total_profit' => $totalProfit,
                'tax_amount' => $totalTaxAmount,
                'product_profit_details' => $productProfitDetails,
                'discount' => $discount,
            ];
        }

        return view('reports.profit_report', compact('profitDetails'));
    }

    public function emiProfitFilter(Request $request)
    {
        $results = InvoiceModel::with(['invoiceItems.products', 'invoiceItems.invoiceItemTaxes'])->where('branch_id', session('branch_id'))
            ->where('oprntl_flag', 'E')->get();
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);


        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if ($start_date && $end_date) {
            $results = $results->filter(function ($invoice) use ($start_date, $end_date) {
                $invoiceDate = Carbon::parse($invoice->invoice_date);
                return $invoiceDate->between($start_date, $end_date);
            });
        }

        $profitDetails = [];

        foreach ($results as $invoice) {
            $totalSellingPrice = 0;
            $totalMarginPrice = 0;
            $totalProfit = 0;
            $productProfitDetails = [];

            foreach ($invoice->invoiceItems as $invoiceItem) {
                $product = $invoiceItem->products;

                $sellingPrice = $invoiceItem->price;
                $marginPrice = $invoiceItem->margin_price;
                $quantity = $invoiceItem->quantity;

                $profit = ($sellingPrice - $marginPrice) * $quantity;

                $taxAmount = 0;
                foreach ($invoiceItem->invoiceItemTaxes as $tax) {
                    $taxAmount += ($tax->tax / 100) * $sellingPrice * $quantity;
                }

                $totalSellingPrice += ($sellingPrice * $quantity) + $taxAmount;

                $totalMarginPrice += $marginPrice * $quantity;

                $totalProfit += $profit;

                $productProfitDetails[] = [
                    'product_name' => $product->name,
                    'selling_price' => $sellingPrice,
                    'margin_price' => $marginPrice,
                    'quantity' => $quantity,
                    'profit' => $profit,
                    'tax_amount' => $taxAmount,
                ];
            }

            $profitDetails[] = [
                'invoice_id' => $invoice->invoice_id,
                'invoice_date' => $invoice->invoice_date,
                'total_selling_price' => $totalSellingPrice,
                'total_margin_price' => $totalMarginPrice,
                'total_profit' => $totalProfit,
                'tax_amount' => $totalSellingPrice - ($totalMarginPrice + $totalProfit),
                'product_profit_details' => $productProfitDetails,

            ];
        }

        return view('reports.profit_report', compact('profitDetails'));

    }
    public function emiProfit(Request $request)
    {
        $results = InvoiceModel::with(['invoiceItems.products', 'invoiceItems.invoiceItemTaxes', 'invoice_emi.invoice_emi_dtls'])->where('branch_id', session('branch_id'))->where('oprntl_flag', 'E')->get();

        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if ($start_date && $end_date) {
            $results = $results->filter(function ($invoice) use ($start_date, $end_date) {
                $invoiceDate = Carbon::parse($invoice->invoice_date);
                return $invoiceDate->between($start_date, $end_date);
            });
        }

        $profitDetails = [];

        foreach ($results as $invoice) {
            $totalSellingPrice = 0;
            $totalMarginPrice = 0;
            $totalProfit = 0;
            $penalty = 0;
            $currentPaidAmount = 0;
            $emiProfitDetails = [];
            $totalDiscountAmount = 0;
            $paidEmiDetails = InvoiceEmiDtlsModel::where('invoice_emi_id', $invoice->invoice_emi->id)
                ->whereIn('oprntl_flag', ['P', 'C'])
                ->get();
            $taxAmount = 0;

            foreach ($invoice->invoiceItems as $invoiceItem) {
                $product = $invoiceItem->products;
                $productSellingPrice = $product->unit_price;
                $productMarginPrice = $product->margin_price;
                $quantity = $invoiceItem->quantity;

                $productProfit = ($productSellingPrice - $productMarginPrice) * $quantity;

                foreach ($invoiceItem->invoiceItemTaxes as $tax) {
                    $taxAmount += ($tax->tax / 100) * ($productSellingPrice * $quantity);
                }

                foreach ($invoice->invoice_emi->invoice_emi_dtls as $emiDetail) {
                    if ($emiDetail->penalty) {
                        $penalty += $emiDetail->penalty;
                    }
                    if ($emiDetail->discount_amount) {
                        $totalDiscountAmount += $emiDetail->discount_amount;
                    }
                }
                $downPayment = $invoice->invoice_emi->down_payment ?? 0;
                $actualAmount = $invoice->invoice_emi->actual_amount ?? 0;
                $emiWithInterest = $actualAmount + $penalty;

                $totalSellingPrice = $downPayment + $emiWithInterest;
                $currentProfit = $downPayment + $currentPaidAmount + $penalty;

                $totalMarginPrice += $productMarginPrice * $quantity;
            }
            foreach ($invoice->invoice_emi->invoice_emi_dtls as $data) {
                $emiProfitDetails[] = [
                    'emi_date' => $data->emi_date,
                    'emi_amount' => $data->emi_amount,
                    'paid_date' => $data->paid_date,
                    'due_date' => $data->due_date,
                    'penalty' => $data->penalty,
                    'discount_amount' => $data->discount_amount,
                    'total_amount' => $data->total_amount,
                ];
            }
            foreach ($paidEmiDetails as $data) {
                if ($data->total_amount) {
                    $currentPaidAmount += $data->total_amount;
                }
            }
            $totalProfit = ($totalSellingPrice - $totalMarginPrice) - $totalDiscountAmount;
            $currentProfit = ($downPayment + $currentPaidAmount) - $totalMarginPrice - $totalDiscountAmount;
            $profitDetails[] = [
                'invoice_id' => $invoice->invoice_id,
                'invoice_date' => $invoice->invoice_date,
                'total_selling_price' => round($totalSellingPrice, 0),
                'total_margin_price' => $totalMarginPrice,
                'total_profit' => round($totalProfit, 0),
                'current_profit' => round($currentProfit, 0),
                'tax_amount' => round($taxAmount, 0),
                'emi_profit_details' => $emiProfitDetails,
            ];
        }

        return view('reports.emi_profit_report', compact('profitDetails'));
    }


    // public function productPurchase(Request $request)
    // {
    //     $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

    //     $results = $isAdmin ? ProductDetailsModel::with('payments', 'paymentStatus', 'branch')->get() : ProductDetailsModel::with('payments', 'paymentStatus', 'branch')->where('branch_id', session('branch_id'))->get();

    //     $request->validate([
    //         'start_date' => 'nullable|date',
    //         'end_date' => 'nullable|date|after_or_equal:start_date',
    //     ]);

    //     $start_date = $request->input('start_date');
    //     $end_date = $request->input('end_date');

    //     if ($start_date && $end_date) {
    //         $start_date = Carbon::parse($start_date);
    //         $end_date = Carbon::parse($end_date);

    //         $results = $results->filter(function ($product) use ($start_date, $end_date) {
    //             return $product->payments->some(function ($payment) use ($start_date, $end_date) {
    //                 $paymentDate = Carbon::parse($payment->payment_date);
    //                 return $paymentDate->between($start_date, $end_date);
    //             });
    //         });
    //     }

    //     $productData = $results->map(function ($product) {
    //         $totalPaidAmount = $product->payments->sum('paid_amount');
    //         $balanceAmount = max(0, $product->total_amount - $totalPaidAmount);

    //         $purchaseDate = $product->payments->first()->payment_date ?? null;

    //         return [
    //             'branch_name' => $product->branch->name ?? '',
    //             'product_name' => $product->name,
    //             'product_code' => $product->code,
    //             'purchase_date' => $purchaseDate,
    //             'purchase_amount' => $product->total_amount,
    //             'paid_amount' => $totalPaidAmount,
    //             'balance_amount' => $balanceAmount,
    //             'supplier_name' => $product->suppliers->supplier_name ?? 'Internal'
    //         ];
    //     });
    //     return view('reports.product_purchase_report', compact('productData'));
    // }
    public function productPurchase(Request $request)
    {
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        $query = PurchaseModel::with(['productDetails.product', 'productDetails.suppliers', 'payments', 'branch']);

        if (!$isAdmin) {
            $query->where('branch_id', session('branch_id'));
        }

        if ($start_date && $end_date) {
            $query->whereBetween('purchase_date', [
                Carbon::parse($start_date)->startOfDay(),
                Carbon::parse($end_date)->endOfDay()
            ]);
        }

        $purchases = $query->get();

        // Prepare data for the view
        $productData = $purchases->map(function ($purchase) {
            $totalPaidAmount = $purchase->payments->sum('paid_amount');
            $balanceAmount = max(0, $purchase->total_amount - $totalPaidAmount);

            return [
                'branch_name' => $purchase->branch->name ?? '',
                'purchase_code' => $purchase->purchase_code,
                'purchase_date' => $purchase->purchase_date,
                'purchase_amount' => $purchase->total_amount,
                'paid_amount' => $totalPaidAmount,
                'balance_amount' => $balanceAmount,
                'supplier_name' => optional($purchase->productDetails->first()?->suppliers)->supplier_name ?? 'Internal',
            ];
        });


        return view('reports.product_purchase_report', compact('productData'));
    }

    public function invoiceEmiFilter(Request $request)
    {
        $results = InvoiceModel::with('branch', 'invoice_emi')->where('branch_id', session('branch_id'))->get();
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'nullable|in:Y,N',
        ]);
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $status = $request->input('status');

        $query = InvoiceModel::query();

        if ($start_date) {
            $query->where('created_at', '>=', $start_date);
        }

        if ($end_date) {
            $query->where('created_at', '<=', $end_date);
        }

        if ($status) {
            $query->where('gst', '=', $status);
        }

        $results = $query->with('branch', 'invoice_emi.invoice_emi_dtls', 'payments', 'discountType')
            ->where('oprntl_flag', 'E')->where('branch_id', session('branch_id'))
            ->get();
        $statusList = DB::table('payment_status')->pluck('name', 'id');
        return view('reports.invoiceEmiReport', compact('results', 'statusList'));
    }

    public function getInvoiceDetails($id)
    {
        $invoice = InvoiceModel::with('invoice_emi.invoice_emi_dtls')->find($id);
        return response()->json([
            'invoice_number' => $invoice->invoice_id,
            'emi_details' => $invoice->invoice_emi->invoice_emi_dtls->map(function ($emiDtl) {
                return [
                    'emi_date' => $emiDtl->emi_date,
                    'emi_amount' => $emiDtl->emi_amount,
                    'paid_date' => $emiDtl->paid_date,
                    // 'paid_amount' => $emiDtl->paid_amount,
                    'due_date' => $emiDtl->due_date,
                    'penalty' => $emiDtl->penalty,
                    'discount_amount' => $emiDtl->discount_amount,
                    'total_amount' => $emiDtl->total_amount,
                ];
            })
        ]);
    }


    public function inventoryReport()
    {
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $product = $isAdmin ? ProductModel::with('category.unitOfMass', 'branch')->where('active_flag', 'A')->get() : ProductModel::with('category.unitOfMass')->where('branch_id', session('branch_id'))->where('active_flag', 'A')->get();
        return view('reports.inventory_report', compact('product'));
    }
    public function gstReport(Request $request)
    {
        // $invoice = InvoiceModel::with('client.user', 'branch', 'discountType', 'invoiceItems.invoiceItemTaxes')->where('gst', 'Y')->where('oprntl_flag', 'A')->where('active_flag', 'A')->get();
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'nullable',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $status = $request->input('status');

        $query = InvoiceModel::query();

        if ($start_date) {
            $query->where('invoice_date', '>=', $start_date);
        }

        if ($end_date) {
            $query->where('invoice_date', '<=', $end_date);
        }

        if ($status) {
            $query->where('status', '=', $status);
        }

        $invoice = $isAdmin ? $query->with('client.user', 'branch', 'discountType', 'invoiceItems.invoiceItemTaxes')->where('gst', 'Y')->where('oprntl_flag', 'A')->where('active_flag', 'A')->get() : $query->with('client.user', 'branch', 'discountType', 'invoiceItems.invoiceItemTaxes')->where('gst', 'Y')->where('branch_id', session('branch_id'))->where('oprntl_flag', 'A')->where('active_flag', 'A')->get();

        $statusList = PaymentStatusModel::pluck('name', 'id');

        return view('reports.gst_report', compact('invoice', 'statusList'));
    }

    public function hsnReport(Request $request)
    {
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'nullable',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $status = $request->input('status');

        $query = InvoiceModel::query();

        if ($start_date) {
            $query->where('invoice_date', '>=', $start_date);
        }

        if ($end_date) {
            $query->where('invoice_date', '<=', $end_date);
        }

        // if ($status) {
        //     $query->where('status', '=', $status);
        // }

        $invoice = $isAdmin ? $query->with('client.user', 'branch', 'discountType', 'invoiceItems.invoiceItemTaxes', 'invoiceItems.products')->where('gst', 'Y')->where('oprntl_flag', 'A')->where('active_flag', 'A')->get() : $query->with('client.user', 'branch', 'discountType', 'invoiceItems.invoiceItemTaxes', 'invoiceItems.products')->where('gst', 'Y')->where('branch_id', session('branch_id'))->where('oprntl_flag', 'A')->where('active_flag', 'A')->get();

        $statusList = PaymentStatusModel::pluck('name', 'id');

        return view('reports.hsn_report', compact('invoice', 'statusList'));
    }
    public function expenseReport(Request $request)
    {
        $isAdmin = in_array(Auth::user()->user_flg, ['SU', 'A']);

        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'status' => 'nullable',
        ]);

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        $category = $request->input('category');

        $query = ExpensesModel::query();

        if ($start_date) {
            $query->where('expense_date', '>=', $start_date);
        }

        if ($end_date) {
            $query->where('expense_date', '<=', $end_date);
        }

        if ($category) {
            $query->where('expense_category_id', '=', $category);
        }

        $expense = $isAdmin ? $query->with('category', 'createdBy', 'updatedBy', 'branch')->where('active_flag', 'A')->get() : $query->with('category', 'createdBy', 'updatedBy', 'branch')->where('branch_id', session('branch_id'))->where('active_flag', 'A')->get();

        $category = ExpenseCategory::pluck('category_name', 'id');

        return view('reports.expenseReport', compact('expense', 'category'));
    }
}
