<?php

namespace App\Http\Controllers;

use App\Models\BranchModel;
use App\Models\ClientAddressModel;
use App\Models\InvoiceModel;
use App\Models\LoyaltyTransactionModel;
use App\Models\OrdersModel;
use App\Models\QuotesModel;
use App\Models\StandardPaperSizeModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Mpdf\Mpdf;


class PDFController extends Controller
{


    public function generateInvoice(Request $request)
    {
        try {

            $addedInvoice = InvoiceModel::with('client.user', 'order.clientAddress', 'branch.company', 'invoiceItems.invoiceItemTaxes', 'payments.mode', 'invoiceItems.products.category.unitOfMass')
                ->where('id', $request->invoice_id)
                ->first();

            if (!$addedInvoice) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            $invoiceItems = $addedInvoice->invoiceItems;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($invoiceItems as $item) {
                $gst = $item->invoiceItemTaxes->sum('tax');
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;
                $products[] = [
                    'name' => $addedInvoice->branch->native_name_flag == 'A' ? ($item->products->native_name ?? $item->products->name) : ($item->products->name ?? ''),
                    'code' => $item->products->hsn_code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->products->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    'gst' => $item->invoiceItemTaxes->sum('tax') ?? 0,
                    'total_price' => $item->quantity * $subtotal,
                    'description' => $item->description ?? '',
                ];
            }

            if ($addedInvoice->discount_type == 1) {
                $discount = $addedInvoice->discount;
            } elseif ($addedInvoice->discount_type == 2) {
                $discount = ($total * $addedInvoice->discount) / 100;
            } else {
                $discount = 0;
            }

            $sumQty = 0;
            foreach ($addedInvoice->invoiceItems as $item) {
                $sumQty += $item->quantity;
            }

            $redeem = LoyaltyTransactionModel::where('client_id', $addedInvoice->client_id)->where('reference_invoice', $addedInvoice->invoice_id)->get();

            $redeemed_points = $redeem->where('type', 'redeem')->sum('points');

            $redeemed_value = $redeemed_points * $addedInvoice->branch->point_redemption;

            $data = [
                'companyName' => $addedInvoice->branch->company->name,
                'companyAddress' => $addedInvoice->branch->company->address,
                'contact_no' => $addedInvoice->branch->company->phone,
                'email' => $addedInvoice->branch->company->email,
                'clientName' => $addedInvoice->client->user->name,
                'clientAddress' => optional(optional($addedInvoice->order)->clientAddress)->address
                    ?? optional(optional($addedInvoice->client)->user)->location
                    ?? '',
                'gst_no' => $addedInvoice->branch->gst_no,
                'invoiceNumber' => $addedInvoice->invoice_id,
                'invoiceDate' => $addedInvoice->invoice_date ? \Carbon\Carbon::parse($addedInvoice->invoice_date)->format('d-m-Y') : '',
                'dueDate' => $addedInvoice->due_date ? \Carbon\Carbon::parse($addedInvoice->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'gst' => $addedInvoice->gst,
                'type' => 'I',
                'payment_terms' => $addedInvoice->payments->isNotEmpty() && $addedInvoice->payments->first()->mode
                    ? $addedInvoice->payments->first()->mode->name
                    : 'N/A',
                'totalQty' => $sumQty,
                'note' => !empty($addedInvoice->note) ? $addedInvoice->note : '',
                'terms' => !empty($addedInvoice->term) ? $addedInvoice->term : $addedInvoice->branch->declaration,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0,
                'settings' => $addedInvoice->branch,
            ];

            $html = view('pdf.others', $data)->render();

            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);

            if ($request->type == 'download') {
                return response($mpdf->Output('invoice.pdf', 'S'))
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'attachment; filename="' . $addedInvoice->invoice_id . '.pdf"');
            }

            if ($request->type == 'print') {
                return response($mpdf->Output('invoice.pdf', 'S'))
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="' . $addedInvoice->invoice_id . '.pdf"');
            }

            return response($mpdf->Output('invoice.pdf', 'S'))
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $addedInvoice->invoice_id . '.pdf"');

        } catch (\Exception $e) {
            Log::error('Error generating invoice: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
    public function generateQuote(Request $request)
    {
        try {

            $addedQuote = QuotesModel::with('client.user', 'branch.company', 'quotesItem.quoteItemTaxes', 'quotesItem.product.category.unitOfMass')
                ->where('id', $request->quote_id)
                ->first();

            if (!$addedQuote) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            $quoteItems = $addedQuote->quotesItem;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($quoteItems as $item) {
                $gst = $item->quoteItemTaxes->sum('tax');
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;
                $products[] = [
                    'name' => $addedQuote->branch->native_name_flag == 'A' ? ($item->product->native_name ?? $item->product->name) : ($item->product->name ?? ''),
                    'code' => $item->product->hsn_code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->product->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    // 'gst' => $item->quoteItemTaxes->sum('tax') ?? 0,
                    'total_price' => $item->quantity * $subtotal,
                ];
            }

            // Calculate the discount
            if ($addedQuote->discount_type == 1) {
                $discount = $addedQuote->discount;
            } elseif ($addedQuote->discount_type == 2) {
                $discount = ($total * $addedQuote->discount) / 100;
            } else {
                $discount = 0;
            }

            $data = [
                'companyName' => $addedQuote->branch->company->name,
                'companyAddress' => $addedQuote->branch->company->address,
                'contact_no' => $addedQuote->branch->company->phone,
                'email' => $addedQuote->branch->company->email,
                'clientName' => $addedQuote->client->user->name,
                'clientAddress' => $addedQuote->client->user->location ?? '',
                'gst_no' => $addedQuote->branch->gst_no,
                'invoiceNumber' => $addedQuote->quote_id,
                'invoiceDate' => $addedQuote->quote_date ? \Carbon\Carbon::parse($addedQuote->quote_date)->format('d-m-Y') : '',
                'dueDate' => $addedQuote->due_date ? \Carbon\Carbon::parse($addedQuote->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'type' => 'Q',
                'settings' => $addedQuote->branch,
            ];

            // Render the HTML
            $html = view('pdf.others', $data)->render();

            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);

            if ($request->type == 'download') {
                return response($mpdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'attachment; filename="' . $addedQuote->quote_id . '.pdf"');
            }

            if ($request->type == 'print') {
                return response($mpdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="' . $addedQuote->quote_id . '.pdf"');
            }

            return response($mpdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $addedQuote->quote_id . '.pdf"');


        } catch (\Exception $e) {

            Log::info('Error in Quote generate' . $e->getMessage());

            return response()->json(['error' => 'Something went wrong. Please try again later.'], 500);
        }
    }
    public function generateOrder(Request $request)
    {
        try {
            $addedOrder = OrdersModel::with('client.user', 'branch.company', 'clientAddress', 'ordersItem.product.category.unitOfMass')
                ->where('id', $request->quote_id)
                ->first();

            if (!$addedOrder) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            $orderItems = $addedOrder->ordersItem;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($orderItems as $item) {
                $gst = 0;
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;
                $products[] = [
                    'name' => $addedOrder->branch->native_name_flag == 'A' ? ($item->product->native_name ?? $item->product->name) : ($item->product->name ?? ''),
                    'code' => $item->product->hsn_code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->product->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    // 'gst' => $item->quoteItemTaxes->sum('tax') ?? 0,
                    'total_price' => $item->quantity * $subtotal,
                ];
            }

            // Calculate the discount
            if ($addedOrder->discount_type == 1) {
                $discount = $addedOrder->discount;
            } elseif ($addedOrder->discount_type == 2) {
                $discount = ($total * $addedOrder->discount) / 100;
            } else {
                $discount = 0;
            }

            $address = $addedOrder->clientAddress->address ?? $addedOrder->client->user->location;
            $data = [
                'companyName' => $addedOrder->branch->company->name,
                'companyAddress' => $addedOrder->branch->company->address,
                'contact_no' => $addedOrder->branch->company->phone,
                'email' => $addedOrder->branch->company->email,
                'clientName' => $addedOrder->client->user->name,
                'clientAddress' => $address,
                'gst_no' => $addedOrder->branch->gst_no,
                'invoiceNumber' => $addedOrder->order_id,
                'invoiceDate' => $addedOrder->order_date ? \Carbon\Carbon::parse($addedOrder->order_date)->format('d-m-Y') : '',
                'dueDate' => $addedOrder->due_date ? \Carbon\Carbon::parse($addedOrder->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'type' => 'O',
                'settings' => $addedOrder->branch,
            ];

            // Render the HTML
            $html = view('pdf.others', $data)->render();

            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);

            if ($request->type == 'download') {
                return response($mpdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'attachment; filename="' . $orderItems->order_id . '.pdf"');
            }

            if ($request->type == 'print') {
                return response($mpdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="' . $orderItems->order_id . '.pdf"');
            }

            return response($mpdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $orderItems->order_id . '.pdf"');


        } catch (\Exception $e) {

            Log::info('Error in Order generate' . $e->getMessage());

            return response()->json(['error' => 'Something went wrong. Please try again later.'], 500);
        }
    }

    public function openQoutePdfPage(Request $request, $id)
    {
        $addedQuote = QuotesModel::with('client.user', 'client.clientAddress', 'branch.company', 'quotesItem.quoteItemTaxes', 'quotesItem.product.category.unitOfMass')
            ->where('id', $id)
            ->first();
        $settings = BranchModel::where('id', $addedQuote->branch_id)->first();
        $paper = StandardPaperSizeModel::where('oprntl_flag', 'A')->where('view_flag', 'A')->get();
        $paper_size = $paper->where('id', $request->paperId)->first();

        // Keep your original return logic unchanged
        if (isset($paper_size) ? $paper_size->name == 'Others' : $settings->paper->name == 'Others') {
            return view('quotes.pdf.quote-generate-pdf', [
                'addedQuote' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'Thermal Printer' : $settings->paper->name == 'Thermal Printer') {
            return view('quotes.pdf.pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A4' : $settings->paper->name == 'A4') {
            return view('quotes.pdf.a4Pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A3' : $settings->paper->name == 'A3') {
            return view('quotes.pdf.a3Pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        }
    }
    public function openOrderPdfPage(Request $request, $id)
    {
        $addedOrder = OrdersModel::with('client.user', 'client.clientAddress', 'clientAddress', 'branch.company', 'ordersItem.product.category.unitOfMass')
            ->where('id', $id)
            ->first();
        $settings = BranchModel::where('id', $addedOrder->branch_id)->first();
        $paper = StandardPaperSizeModel::where('oprntl_flag', 'A')->where('view_flag', 'A')->get();
        $paper_size = $paper->where('id', $request->paperId)->first();

        // Keep your original return logic unchanged
        if (isset($paper_size) ? $paper_size->name == 'Others' : $settings->paper->name == 'Others') {
            return view('orders.pdf.order-generate-pdf', [
                'addedQuote' => $addedOrder,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'Thermal Printer' : $settings->paper->name == 'Thermal Printer') {
            return view('orders.pdf.pdf', [
                'addedInvoice' => $addedOrder,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A4' : $settings->paper->name == 'A4') {
            return view('orders.pdf.a4Pdf', [
                'addedInvoice' => $addedOrder,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A3' : $settings->paper->name == 'A3') {
            return view('orders.pdf.a3Pdf', [
                'addedInvoice' => $addedOrder,
                'settings' => $settings,
                'paper' => $paper
            ]);
        }
    }

    public function generatePdf(Request $request, $id)
    {
        $addedInvoice = InvoiceModel::with('client.user', 'order.clientAddress', 'branch.company', 'invoiceItems.invoiceItemTaxes', 'payments.mode', 'invoiceItems.products.category.unitOfMass')
            ->where('id', $id)
            ->first();

        $invoiceController = new InvoiceController();
        $dueAmount = $invoiceController->dueAmount($addedInvoice->client);

        $settings = BranchModel::with('paper')->where('id', $addedInvoice->branch_id)->first();

        $redeem = LoyaltyTransactionModel::where('client_id', $addedInvoice->client_id)->where('reference_invoice', $addedInvoice->invoice_id)->get();

        $redeemed_points = $redeem->where('type', 'redeem')->sum('points');

        $redeemed_value = $redeemed_points * $settings->point_redemption;

        $paper = StandardPaperSizeModel::where('oprntl_flag', 'A')->where('view_flag', 'A')->get();

        $paper_size = $paper->where('id', $request->paperId)->first();

        if (isset($paper_size) ? $paper_size->name == 'Others' : $settings->paper->name == 'Others') {

            return view('pdf.billGenerate', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'Thermal Printer' : $settings->paper->name == 'Thermal Printer') {

            return view('pdf.pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper,
                'dueAmount' => $dueAmount,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0

            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A4' : $settings->paper->name == 'A4') {

            return view('pdf.a4Pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0

            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A3' : $settings->paper->name == 'A3') {

            return view('pdf.a3Pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0

            ]);
        }

    }

}