<?php

namespace App\Http\Controllers;

use App\Mail\InvoiceCreated;
use App\Models\BranchModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\InvoiceItemModel;
use App\Models\InvoiceItemTaxModel;
use App\Models\InvoiceModel;
use App\Models\LoyaltyTransactionModel;
use App\Models\PaymentsModel;
use App\Models\PaymentStatusModel;
use App\Models\ProductModel;
use App\Models\QuotesItemsModel;
use App\Models\QuotesModel;
use App\Models\RightsModel;
use App\Models\User;
use App\Models\TaxesModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Mpdf\Mpdf;

class QuotesController extends Controller
{
    public function index(Request $request)
    {
        $quotes = QuotesModel::with('client.user')->where('branch_id', session('branch_id'))->orderBy('id', 'desc')->get();

        return view('quotes.index', compact('quotes'));
    }
    public function add()
    {
        $branch_id = session('branch_id');

        $clients = ClientModel::with('user')->get();
        $products = ProductModel::with('branch')->where('branch_id', $branch_id)->whereIn('product_type', [2, 3])->get();

        $discount_types = DB::table('discount_type')->get();
        $taxes = TaxesModel::where('is_default', 1)->where('branch_id', $branch_id)->get();

        return view('quotes.add', compact('clients', 'products', 'discount_types', 'taxes'));
    }

    public function store(Request $request)
    {

        $branch_id = session('branch_id');
        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();
        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = saveClient($request);

            if (!$client_id) {
                return redirect()->back()->with('error', 'Failed to create client');
            }

            $lastQuotes = QuotesModel::orderBy('id', 'desc')->first();
            $quote_id = $this->generateQuoteId($lastQuotes);

            $quote = QuotesModel::create([
                'quote_id' => $quote_id,
                'client_id' => $client_id,
                'quote_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $branch_id,
                'status' => $request->status ?? '',
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                QuotesItemsModel::create([
                    'quote_id' => $quote->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            DB::commit();

            return redirect()->route('quote-pdf', ['id' => $quote->id])->with('success', 'Quote created successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Add  Quotes' . $e->getMessage());

            return redirect()->back()->with('error', 'Failed to add quote: ' . $e->getMessage())->withInput();
        }
    }


    public function edit($id, $type)
    {
        $branch_id = session('branch_id');

        $clients = ClientModel::with('user')->get();

        $products = ProductModel::with('branch')->where('branch_id', $branch_id)->whereIn('product_type', [2, 3])->get();

        $discount_types = DB::table('discount_type')->get();

        $quote = QuotesModel::with('quotesItem', 'client.user')->findOrFail($id);

        $branch = BranchModel::where('id', session('branch_id'))->first();

        $default_taxes = TaxesModel::where('is_default', 1)->where('branch_id', session('branch_id'))->pluck('id')->toArray();

        $all_taxes = TaxesModel::where('show_flg', 'Y')->where('branch_id', session('branch_id'))->get();

        if ($type == 'convert') {
            return view('quotes.convert', compact('clients', 'products', 'discount_types', 'quote', 'type', 'branch', 'all_taxes', 'default_taxes'));
        }
        return view('quotes.edit', compact('clients', 'products', 'discount_types', 'quote', 'type', 'branch'));

    }

    public function update(Request $request)
    {
        if ($request->type == 'convert') {
            return $this->processConvert($request);
        } else {


            $request->validate([
                'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
                'dynamicFields.*.unit_price' => 'required|numeric',
                'dynamicFields.*.amount' => 'required|numeric',
            ]);
            try {
                $quote = QuotesModel::findOrFail($request->id);

                $quote->update([
                    'client_id' => $request->client_id,
                    'quote_date' => $request->quote_date,
                    'due_date' => $request->due_date,
                    'amount' => $request->sub_total,
                    'final_amount' => $request->total_amount,
                    'discount_type' => $request->discount_type ?? null,
                    'discount' => $request->discount ?? null,
                ]);

                $productIds = [];

                foreach ($request->products as $field) {
                    $product = ProductModel::find($field['id']);

                    $quoteItem = QuotesItemsModel::updateOrCreate(
                        [
                            'quote_id' => $quote->id,
                            'product_id' => $field['id'],
                        ],
                        [
                            'product_name' => $product->name ?? '',
                            'available_quantity' => $product->qty,
                            'quantity' => $field['qty'],
                            'price' => $field['unit_price'],
                            'total' => $field['amount'],
                        ]
                    );

                    $productIds[] = $field['id'];
                }

                // Remove deleted items
                QuotesItemsModel::where('quote_id', $quote->id)
                    ->whereNotIn('product_id', $productIds)
                    ->delete();

                return redirect()->route('quotes-index')
                    ->with('success', 'Quote updated successfully');
            } catch (\Exception $e) {

                DB::rollBack();

                \Log::info('While Update  Quotes' . $e->getMessage());

                return redirect()->back()->with('error', 'Failed to update quote: ' . $e->getMessage())->withInput();
            }
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            $quote = QuotesModel::with('quotesItem.quoteItemTaxes')->findOrFail($id);

            $quote->quotesItem->each(function ($item) {
                $item->quoteItemTaxes()->delete();
            });

            $quote->quotesItem()->delete();
            $quote->delete();

            DB::commit();

            return redirect()->route('quotes-index')->with('success', 'Quote deleted successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Deleting Quote' . $e->getMessage());

            return redirect()->route('quotes-index')->with('error', 'An error occurred while deleting the quote');
        }
    }

    public function getClientAddress($phone)
    {
        $user_id = User::where('phone', $phone)->value('id');
        $client = ClientModel::with('clientAddress')->where('user_id', $user_id)->value('id');
        $clientID = isset($client) ? $client : $phone;
        $client_address = ClientAddressModel::where('client_id', $clientID)->get();

        return response()->json([
            'address' => $client_address ?? '',
        ]);
    }

    public function storeClientAddress(Request $request)
    {
        $clientNumber = is_array($request->client_id) ? $request->client_id[0] : $request->client_id;

        $clientID = ClientModel::where('user_id', User::where('phone', $clientNumber)->value('id'))->value('id');

        $validated = $request->validate([
            'address' => 'required|string|max:255',
        ]);

        ClientAddressModel::where('client_id', $clientID)
            ->update(['oprtnl_flag' => 'I']);

        ClientAddressModel::create([
            'client_id' => $clientID,
            'address' => $validated['address'],
            'branch_id' => session('branch_id'),
            'created_by' => Auth::user()->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Address saved successfully.',
        ]);
    }

    public function removeClientAddress(Request $request)
    {
        if ($request->addressID && $request->clientID) {
            ClientAddressModel::where('client_id', $request->clientID)
                ->where('id', $request->addressID)
                ->delete();

            return response()->json(['success' => true, 'message' => 'Address removed successfully']);
        }

        return response()->json(['success' => false, 'message' => 'Invalid request'], 400);
    }


    public function processConvert($request)
    {

        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
        ]);

        if (($request->cash + $request->upi + $request->online) > $request->total_amount) {
            return back()->with('error', 'The total of cash, UPI, and online payments cannot exceed the total amount.');
        }

        try {
            DB::beginTransaction();

            $quote = QuotesModel::findOrFail($request->id);
            $invoice_id = getPrefixes($request, $quote->client_id);

            $loyalty = BranchModel::where('id', session('branch_id'))->first();

            $clientPoints = ClientModel::where('id', $quote->client_id)->first();

            $loyalty_amount = $loyalty->loyalty_amount;
            $loyalty_config = $loyalty->loyalty_config;

            if (!empty($request->redeem_points)) {
                $current_point = $request->available_point - $request->redeem_points;

                LoyaltyTransactionModel::create([
                    'client_id' => $quote->client_id,
                    'points' => $request->redeem_points,
                    'type' => 'redeem',
                    'reference_invoice' => $invoice_id,
                    'created_by' => Auth::user()->id,
                ]);
                $clientPoints->update([
                    'loyalty_points' => $current_point,
                ]);

            }

            if ($loyalty_config == 'A') {
                $earned_points = floor($request->total_amount / $loyalty_amount);
                $points = $clientPoints->loyalty_points + $earned_points;
                $clientPoints->update([
                    'loyalty_points' => $points,
                ]);

                LoyaltyTransactionModel::create([
                    'client_id' => $quote->client_id,
                    'points' => $earned_points,
                    'type' => 'earn',
                    'reference_invoice' => $invoice_id,
                    'created_by' => Auth::user()->id,
                ]);

            }

            $paid_amount = $request->upi + $request->cash + $request->online;
            if ($paid_amount == 0) {
                $status = 'Unpaid';
            } elseif ($paid_amount == $request->total_amount) {
                $status = 'Paid';
            } else {
                $status = 'Partially Paid';
            }
            $status_id = PaymentStatusModel::where('name', $status)->value('id');


            $invoice = InvoiceModel::create([
                'invoice_id' => $invoice_id,
                'client_id' => $quote->client_id,
                'invoice_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type,
                'discount' => $request->discount,
                'note' => $request->note,
                'term' => $request->term,
                'template_id' => $request->template_id,
                'branch_id' => session('branch_id'),
                'gst' => $request->gst,
                'quote_id' => $quote->quote_id,
                'status' => $status_id,
                'created_by' => Auth::user()->id,
            ]);

            // $products = [];
            foreach ($request->products as $field) {
                $product = ProductModel::findOrFail($field['id']);
                $available_qty = $product->qty - $field['qty'];

                $product->update(['qty' => $available_qty]);

                $item = InvoiceItemModel::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $field['id'],
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                    'product_name' => $product->name,
                ]);
                if ($request->gst == "Y") {
                    foreach ($field['tax'] as $taxId) {
                        InvoiceItemTaxModel::create([
                            'invoice_item_id' => $item->id,
                            'tax_id' => $taxId,
                            'tax' => TaxesModel::find($taxId)->value,
                        ]);
                    }
                }
            }
            if ($status_id == '2' || $status_id == '3') {
                if (!empty($request->upi)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->upi,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 1
                    ]);
                }
                if (!empty($request->cash)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->cash,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 3
                    ]);
                }
                if (!empty($request->online)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->online,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 2
                    ]);
                }
            }
            $quote->update([
                'oprntl_flag' => 'C',
                'status' => $request->status ?? 2,
            ]);

            DB::commit();
            return redirect()->route('quotes-index')->with('success', 'Quotes converted to invoice successfully');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('While converting Quotes to Invoice: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to convert quotes: ')->withInput();
        }
    }

    protected function generateQuoteId($lastQuotes)
    {
        if ($lastQuotes) {
            $lastQuotesId = $lastQuotes->quote_id;
            $parts = explode('-', $lastQuotesId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSQOT-001';
    }
}