<?php

namespace App\Imports;

use App\Models\CategoryModel;
use App\Models\PaymentStatusModel;
use App\Models\ProductModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductPaymentModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductTypeModel;
use App\Models\SupplierModel;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use Maatwebsite\Excel\Concerns\RemembersRowNumber;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Events\AfterImport;
use Illuminate\Support\Facades\Auth;

class ProductImport implements ToModel, WithHeadingRow, WithEvents
{
    use RemembersRowNumber;

    private $row;

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function (BeforeImport $event) {
                DB::beginTransaction();
            },
            AfterImport::class => function (AfterImport $event) {
                $errors = session()->get('pro_err', []);
                if (empty($errors)) {
                    DB::commit();
                } else {
                    DB::rollBack();
                }
            }
        ];
    }

    public function model(array $row)
    {
        $this->row = $this->getRowNumber();
        $hasErrors = false;

        if (empty($row['product_name'])) {
            $this->storeErrors('Product Name', $row['product_name'] ?? '', 'Enter Product Name');
            $hasErrors = true;
        }

        if (empty($row['product_code'])) {
            $this->storeErrors('Product Code', $row['product_code'] ?? '', 'Enter Product Code');
            $hasErrors = true;
        } else {

            $existingProduct = ProductModel::where('code', $row['product_code'])
                ->where('branch_id', session('branch_id'))
                ->first();

            if ($existingProduct) {
                $this->storeErrors('Product Code', $row['product_code'], 'Duplicate Product Code for this branch');
                $hasErrors = true;
            }
        }


        if (!$this->isValidNumeric($row['sellingpriceperunit'])) {
            $this->storeErrors('Selling Price', $row['sellingpriceperunit'] ?? '', 'Enter valid Selling Price');
            $hasErrors = true;
        } elseif (empty($row['sellingpriceperunit'])) {
            $this->storeErrors('Selling Price', $row['sellingpriceperunit'] ?? '', 'Enter Selling Price');
            $hasErrors = true;
        }

        if (!$this->isValidNumeric($row['marginpriceperunit'])) {
            $this->storeErrors('Buying Price', $row['marginpriceperunit'] ?? '', 'Enter valid Buying Price');
            $hasErrors = true;
        } elseif (empty($row['marginpriceperunit'])) {
            $this->storeErrors('Buying Price', $row['marginpriceperunit'] ?? '', 'Enter Buying Price');
            $hasErrors = true;
        }

        if (!$this->isValidNumeric($row['quantity'])) {
            $this->storeErrors('Quantity', $row['quantity'] ?? '', 'Enter valid Quantity');
            $hasErrors = true;
        } elseif (empty($row['quantity'])) {
            $this->storeErrors('Quantity', $row['quantity'] ?? '', 'Enter  Quantity');
            $hasErrors = true;
        }

        $categoryId = $this->getCategory($row['categoryname']);
        if (!$categoryId) {
            $hasErrors = true;
        }

        $supplierId = $this->getSupplier($row['suppliername']);
        if (!$supplierId) {
            $hasErrors = true;
        }

        $productTypeId = $this->getProductType($row['producttype']);
        if (!$productTypeId) {
            $hasErrors = true;
        }

        if ($hasErrors) {
            return null;
        }

        try {
            $product = ProductModel::where('code', $row['product_code'])->first();
            if ($product) {
                return $this->updateProduct($product, $row);
            } else {
                return $this->createProduct($row);
            }
        } catch (\Exception $e) {

            \Log::error('While importing product at row ' . $this->row . ': ' . $e->getMessage());
            return null;
        }
    }

    protected function updateProduct($product, $row)
    {
        $total = (float) $row['quantity'] * (float) $row['marginpriceperunit'];
        $total_amount = $total ?? 0;

        $product->update([
            'qty' => $product->qty + $row['quantity'],
            'unit_price' => $row['sellingpriceperunit'],
            'margin_price' => $row['marginpriceperunit'] ?? 0,
            'product_type' => $this->getProductType($row['producttype']),
        ]);

        $productDetail = $this->createProductDetails($product, $row, $total_amount);
        $this->processPayments($productDetail, $row, $total_amount);
        $this->createProductHistory($product, $row);

        return $product;
    }

    protected function createProduct($row)
    {
        $total = (float) $row['quantity'] * (float) $row['marginpriceperunit'];
        $total_amount = $total ?? 0;
        $product = new ProductModel([
            'name' => $row['product_name'],
            'native_name' => $row['native_name'],
            'code' => $row['product_code'],
            'hsn_code' => $row['hsn_code'],
            'category_id' => $this->getCategory($row['categoryname']),
            'supplier_id' => $this->getSupplier($row['suppliername']),
            'unit_price' => $row['sellingpriceperunit'],
            'description' => $row['description'] ?? null,
            'branch_id' => session('branch_id') ?? null,
            'margin_price' => $row['marginpriceperunit'] ?? 0,
            'qty' => $row['quantity'],
            'total_qty' => $row['quantity'],
            'product_type' => $this->getProductType($row['producttype']),
        ]);

        $product->save();

        $productDetail = $this->createProductDetails($product, $row, $total_amount);
        $this->processPayments($productDetail, $row, $total_amount);
        $this->createProductHistory($product, $row);

        return $product;
    }

    protected function createProductDetails($product, $row, $total_amount)
    {
        $paid_amount = ($row['upi'] ?? 0) + ($row['cash'] ?? 0) + ($row['online'] ?? 0);

        if ($paid_amount == 0) {
            $status = 'Unpaid';
        } elseif ($paid_amount >= $total_amount) {
            $status = 'Paid';
        } else {
            $status = 'Partially Paid';
        }

        $status_id = PaymentStatusModel::where('name', $status)->value('id');

        if (!$status_id) {
            $status_id = 1 ?? null;
        }

        return ProductDetailsModel::create([
            'product_id' => $product->id,
            'product_details_code' => $this->generateProductDetailCode(),
            'name' => $row['product_name'],
            'code' => $row['product_code'],
            'category_id' => $this->getCategory($row['categoryname']),
            'supplier_id' => $this->getSupplier($row['suppliername']),
            'unit_price' => $row['sellingpriceperunit'],
            'margin_price' => $row['marginpriceperunit'] ?? 0,
            'description' => $row['description'] ?? null,
            'branch_id' => session('branch_id'),
            'qty' => $row['quantity'],
            'total_amount' => $total_amount,
            'status' => $status_id,
            'paid_amount' => $paid_amount,
        ]);
    }

    protected function createProductHistory($product, $row)
    {
        ProductHistoryModel::create([
            'product_id' => $product->id,
            'branch_id' => session('branch_id'),
            'name' => $row['product_name'],
            'native_name' => $row['native_name'],
            'code' => $row['product_code'],
            'category_id' => $this->getCategory($row['categoryname']),
            'supplier_id' => $this->getSupplier($row['suppliername']),
            'unit_price' => $row['sellingpriceperunit'],
            'margin_price' => $product->margin_price,
            'qty' => $row['quantity'],
            'description' => $row['description'] ?? null,
            'action' => 'A',
            'created_by' => Auth::id(),
        ]);
    }

    protected function processPayments($productDetail, $row, $total_amount)
    {
        $paymentTypes = ['upi', 'online', 'cash'];
        foreach ($paymentTypes as $paymentType) {
            if (!empty($row[$paymentType])) {
                ProductPaymentModel::create([
                    'product_dtl_id' => $productDetail->product_details_sid,
                    'payment_type' => $paymentType,
                    'payment_date' => isset($row['paymentdate']) ? Date::excelToDateTimeObject($row['paymentdate']) : now(),
                    'total_amount' => $total_amount,
                    'paid_amount' => $row[$paymentType],
                ]);
            }
        }
    }

    protected function getCategory($data)
    {
        $category = CategoryModel::where('name', $data)->pluck('id')->first();
        if (empty($category)) {
            $this->storeErrors('Category', $data);
            return false;
        }
        return $category;
    }

    protected function getSupplier($data)
    {
        $supplier = SupplierModel::where('supplier_name', $data)->pluck('id')->first();
        if (empty($supplier)) {
            $this->storeErrors('Supplier', $data);
            return false;
        }
        return $supplier;
    }

    protected function getProductType($data)
    {
        $productType = ProductTypeModel::where('name', $data)->pluck('id')->first();
        if (empty($productType)) {
            $this->storeErrors('ProductType', $data);
            return false;
        }
        return $productType;
    }

    protected function isValidNumeric($data)
    {
        return !empty($data) && is_numeric($data);
    }

    private function storeErrors($key, $value = ' ', $error = 'Please check the spelling')
    {
        $errors = session()->get('pro_err', []);
        $errors[] = htmlspecialchars($key) . ': ' . htmlspecialchars($value) . ' ' . htmlspecialchars($error) . ' at row no: ' . $this->row;
        session()->put('pro_err', $errors);
    }

    private function generateProductDetailCode(): string
    {
        $lastDetail = ProductDetailsModel::orderBy('product_details_sid', 'desc')->first();

        if ($lastDetail && $lastDetail->product_details_code) {
            $lastCode = $lastDetail->product_details_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PROD-001';
    }

}